<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use App\Models\TaskRequest;
use App\Models\UserLedger;
use Carbon\Carbon;

class SalaryController extends Controller
{
    /**
     * Display weekly salary page with member wallet logic
     */
    public function index()
    {
        $user = Auth::user();
        $startOfWeek = Carbon::now()->startOfWeek(); 
        $endOfWeek = Carbon::now()->endOfWeek();

        // 1. Total Approved Referrals since Monday
        $totalReferrals = User::where('ref_by', $user->ref_id)
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->whereHas('deposits', function($query) {
                $query->where('status', 'approved');
            })
            ->count();

        // 2. Member requirements per task
        $tasksData = [1 => 3, 2 => 6, 3 => 10, 4 => 15, 5 => 30, 6 => 45];

        // 3. Subtract members already used in claimed rewards
        $claimedTaskIds = TaskRequest::where('user_id', $user->id)
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->pluck('task_id')
            ->toArray();

        $membersUsed = 0;
        foreach ($claimedTaskIds as $tid) {
            if (isset($tasksData[$tid])) {
                $membersUsed += $tasksData[$tid];
            }
        }

        // 4. Remaining Member Balance
        $newReferralsCount = $totalReferrals - $membersUsed;
        if($newReferralsCount < 0) $newReferralsCount = 0;

        return view('app.main.salary.index', compact('user', 'newReferralsCount', 'claimedTaskIds'));
    }

    /**
     * Deduct points and claim reward
     */
    public function claim($id)
    {
        $user = Auth::user();
        $tasks = [
            1 => ['invites' => 3, 'bonus' => 100],
            2 => ['invites' => 6, 'bonus' => 200],
            3 => ['invites' => 10, 'bonus' => 400],
            4 => ['invites' => 15, 'bonus' => 600],
            5 => ['invites' => 30, 'bonus' => 1600],
            6 => ['invites' => 45, 'bonus' => 2500],
        ];

        if (!isset($tasks[$id])) {
            return back()->with('error', 'Invalid Task Level.');
        }

        $startOfWeek = Carbon::now()->startOfWeek();
        $endOfWeek = Carbon::now()->endOfWeek();

        // Check if level already claimed
        $alreadyClaimed = TaskRequest::where('user_id', $user->id)
            ->where('task_id', $id)
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->exists();

        if ($alreadyClaimed) {
            return back()->with('error', 'Reward already claimed!');
        }

        // Deduction Logic
        $totalReferrals = User::where('ref_by', $user->ref_id)
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->whereHas('deposits', function($query) {
                $query->where('status', 'approved');
            })
            ->count();

        $claimedTasks = TaskRequest::where('user_id', $user->id)
            ->whereBetween('created_at', [$startOfWeek, $endOfWeek])
            ->pluck('task_id')->toArray();

        $usedMembers = 0;
        foreach ($claimedTasks as $tid) {
            if (isset($tasks[$tid])) { $usedMembers += $tasks[$tid]['invites']; }
        }

        $availableBalance = $totalReferrals - $usedMembers;

        if ($availableBalance < $tasks[$id]['invites']) {
            return back()->with('error', "Insufficient member points.");
        }

        // Add Bonus & Record
        $user->increment('balance', $tasks[$id]['bonus']);

        TaskRequest::create([
            'user_id' => $user->id,
            'task_id' => $id,
            'bonus' => $tasks[$id]['bonus'],
            'status' => '1', 
            'created_at' => now(),
        ]);

        UserLedger::create([
            'user_id' => $user->id,
            'amount' => $tasks[$id]['bonus'],
            'type' => 'credit',
            'reason' => 'weekly_salary',
            'description' => "Level $id Reward. Used ".$tasks[$id]['invites']." points.",
            'date' => date('d-m-Y H:i')
        ]);

        return back()->with('success', 'Reward Rs ' . $tasks[$id]['bonus'] . ' claimed!');
    }
}
