<?php

namespace App\Http\Controllers\user;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Rebate;
use App\Models\Task;
use App\Models\TaskRequest;
use App\Models\User;
use App\Models\Purchase;
use App\Models\UserLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;

class MiningController extends Controller
{
    /**
     * Show running mining orders
     */
    public function running_mining()
    {
        return view('app.main.order.order');
    }

    /**
     * Start mining for a specific package
     */
    public function start_mining(int $pack_id)
    {
        $purchase = Purchase::where('package_id', $pack_id)
            ->where('user_id', auth()->id())
            ->where('status', 'active')
            ->orderByDesc('id')
            ->first();

        if ($purchase) {
            $purchase->running_status = 'running';
            $purchase->save();
        }

        return back();
    }

    /**
     * Claim daily income (POST)
     */
    public function received_amount(Request $request)
    {
        $user = Auth::user();

        if ($user->receive_able_amount <= 0) {
            return response()->json([
                'status' => false,
                'error' => 'Income not added yet.'
            ]);
        }

        $amount = $user->receive_able_amount;
        $user->balance += $amount;
        $user->receive_able_amount = 0;
        $user->save();

        // FIXED: Changed to credit and added type
        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'daily_income',
            'perticulation' => 'Daily commission received.',
            'amount' => $amount,
            'credit' => $amount,
            'type' => 'credit',
            'status' => 'approved',
            'date' => now()
        ]);

        return response()->json([
            'status' => true,
            'success' => 'My Commission Received.',
            'received_balance' => price($user->receive_able_amount),
            'balance' => price($user->balance)
        ]);
    }

    /**
     * Claim referral reward
     */
    public function received_target_registered()
    {
        $user = Auth::user();
        $first_level_count = User::where('ref_by', $user->ref_id)->count();

        if ($first_level_count < setting('total_member_register_reword')) {
            return response()->json([
                'status' => false,
                'message' => 'Not yet ready for receiving reward.',
                'reword' => price($user->reword_balance)
            ]);
        }

        if ($user->reword_balance <= 0) {
            return response()->json([
                'status' => false,
                'message' => 'Reward already claimed.',
                'reword' => price($user->reword_balance)
            ]);
        }

        $amount = $user->reword_balance;
        $user->balance += $amount;
        $user->reword_balance = 0;
        $user->save();

        // FIXED: Changed 'debit' to 'credit' and added 'type'
        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'reword',
            'perticulation' => 'Referral registration reward received.',
            'amount' => $amount,
            'credit' => $amount,
            'type' => 'credit',
            'status' => 'approved',
            'date' => now()
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Congratulations! You have received your referral reward.',
            'reword' => price($user->reword_balance)
        ]);
    }

    /**
     * Claim invest commission
     */
    public function received_invest_commission()
    {
        $user = Auth::user();

        if ($user->invest_cumulative_balance <= 0) {
            return response()->json([
                'status' => false,
                'message' => 'Receivable amount not eligible.',
                'invest_balance' => price($user->invest_cumulative_balance)
            ]);
        }

        $amount = $user->invest_cumulative_balance;
        $user->balance += $amount;
        $user->invest_cumulative_balance_received += $amount;
        $user->invest_cumulative_balance = 0;
        $user->save();

        // FIXED: Changed 'debit' to 'credit' and added 'type'
        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'invest_commission',
            'perticulation' => 'Invest commission received.',
            'amount' => $amount,
            'credit' => $amount,
            'type' => 'credit',
            'status' => 'approved',
            'date' => now()
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Congratulations! You have received invest commission.',
            'invest_balance' => price($user->invest_cumulative_balance),
            'cumulative' => price($user->invest_cumulative_balance_received)
        ]);
    }

    /**
     * Claim interest commission
     */
    public function received_interest_commission()
    {
        $user = Auth::user();

        if ($user->interest_cumulative_balance <= 0) {
            return response()->json([
                'status' => false,
                'message' => 'Receivable amount not eligible.',
                'interest_receive_balance' => price($user->interest_cumulative_balance)
            ]);
        }

        $amount = $user->interest_cumulative_balance;
        $user->balance += $amount;
        $user->interest_cumulative_balance_received += $amount;
        $user->interest_cumulative_balance = 0;
        $user->save();

        // FIXED: Changed 'debit' to 'credit' and added 'type'
        UserLedger::create([
            'user_id' => $user->id,
            'reason' => 'interest_commission',
            'perticulation' => 'Interest commission received.',
            'amount' => $amount,
            'credit' => $amount,
            'type' => 'credit',
            'status' => 'approved',
            'date' => now()
        ]);

        return response()->json([
            'status' => true,
            'message' => 'Congratulations! You have received interest commission.',
            'interest_receive_balance' => price($user->interest_cumulative_balance),
            'interest_received_balance' => price($user->interest_cumulative_balance_received)
        ]);
    }

    /**
     * Show mining process page
     */
    public function process()
    {
        return view('app.main.order.process');
    }

    /**
     * Apply for a task commission
     */
    public function apply_task_commission(int $task_id)
    {
        $task = Task::find($task_id);
        if (!$task) return back();

        // Check if already submitted
        $submitted = TaskRequest::where('user_id', auth()->id())
            ->where('task_id', $task_id)
            ->where('status', 'pending')
            ->exists();

        if ($submitted) {
            return redirect('home')->with('success', 'Already Submitted.');
        }

        $teamUsers = User::where('ref_by', auth()->user()->ref_id)->get();
        if ($teamUsers->count() < $task->team_size) {
            return redirect('task')->with('error', 'You need more team members.');
        }

        $totalDeposit = Deposit::whereIn('user_id', $teamUsers->pluck('id')->toArray())
            ->where('status', 'approved')
            ->sum('amount');

        if ($totalDeposit < $task->invest) {
            return redirect('task')->with('error', 'Need more investment from team.');
        }

        // Submit task request
        TaskRequest::create([
            'task_id' => $task->id,
            'user_id' => auth()->id(),
            'team_invest' => $task->invest,
            'team_size' => $task->team_size
        ]);

        // FIXED: Changed 'debit' to 'credit' and added 'type'
        UserLedger::create([
            'user_id' => auth()->id(),
            'reason' => 'task',
            'perticulation' => 'Task request submitted.',
            'amount' => $task->bonus,
            'credit' => $task->bonus,
            'type' => 'credit',
            'status' => 'approved',
            'date' => now()
        ]);

        return redirect('task')->with('success', 'Your application has been sent to the owner.');
    }
}
