<?php

namespace App\Http\Controllers\admin;

use App\Http\Controllers\Controller;
use App\Models\Admin;
use App\Models\Recharge;
use App\Models\User;
use App\Models\Purchase;
use App\Models\Package;
use App\Models\UserLedger;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Carbon\Carbon;

class AdminController extends Controller
{
    /**
     * Cron Job function: 
     * Handles Daily plan expiration and Maturity payouts.
     * Automatic credit for Daily plans is REMOVED to prevent double income.
     */
    public function interest()
    {
        // Get all active plans
        $purchases = Purchase::where('status', 'active')->get();

        foreach ($purchases as $purchase) {
            $user = User::find($purchase->user_id);
            $package = Package::find($purchase->package_id);

            if (!$user || !$package) {
                continue;
            }

            $now = Carbon::now();

            // --- DAILY MODE ---
            // Logic removed: Cron no longer adds balance automatically for daily plans.
            // User must click the "Claim" button in the app to receive income.
            if ($package->drop_type === 'daily' || $package->income_mode === 'daily') {
                
                // Check if the plan has expired
                if (Carbon::parse($purchase->validity)->isPast()) {
                    $purchase->status = 'inactive';
                    $purchase->save();
                }
                
                // Note: The $purchase->date update and balance addition 
                // must now happen in your Mining/Claim Controller.
            }

            // --- MATURITY MODE ---
            // Maturity income is usually automatic as it is a one-time payment at the end.
            elseif ($package->drop_type === 'maturity' || $package->income_mode === 'maturity') {

                $validUntil = Carbon::parse($purchase->created_at)->addDays((int)$package->validity);

                if ($now->greaterThanOrEqualTo($validUntil)) {

                    $amount = (double)$package->commission_with_avg_amount;

                    // 1. Add full commission to user's main balance
                    $user->balance = ($user->balance ?? 0) + $amount;
                    $user->receive_able_amount = ($user->receive_able_amount ?? 0) + $amount;
                    $user->save();

                    // 2. TRIGGER TEAM COMMISSION
                    $this->distributeLevelCommission($user, $amount);

                    // 3. Ledger record
                    $ledger = new UserLedger();
                    $ledger->user_id = $user->id;
                    $ledger->reason = 'maturity_income';
                    $ledger->perticulation = $package->name . ' maturity profit released';
                    $ledger->amount = $amount;
                    $ledger->credit = $amount;
                    $ledger->type = 'credit';
                    $ledger->status = 'approved';
                    $ledger->date = now();
                    $ledger->save();

                    // 4. Mark purchase as inactive
                    $purchase->status = 'inactive';
                    $purchase->save();
                }
            }
        }

        return response()->json([
            'status' => 'ok', 
            'message' => 'Cron processed: Validity checked and Maturity released. Daily income remains for manual claim.'
        ]);
    }

    /**
     * TEAM COMMISSION LOGIC (3 Levels)
     * Distributes 10%, 5%, and 2% to uplines who have an active package
     */
    public function distributeLevelCommission($user, $amount)
    {
        $levels = [
            1 => 10, // Level 1
            2 => 5,  // Level 2
            3 => 2,  // Level 3
        ];

        $currentUser = $user;

        foreach ($levels as $level => $percent) {
            
            if (!$currentUser->ref_by) {
                break;
            }

            $upline = User::where('ref_id', $currentUser->ref_by)->first();
            if (!$upline) {
                break;
            }

            // Check if Upline has an active purchase
            $uplinePurchase = Purchase::where('user_id', $upline->id)
                                      ->where('status', 'active')
                                      ->first();

            if (!$uplinePurchase) {
                $currentUser = $upline; 
                continue;
            }

            $commission = ($amount * $percent) / 100;

            $upline->increment('balance', $commission);

            UserLedger::create([
                'user_id' => $upline->id,
                'reason' => 'rebate_income',
                'perticulation' => "Level {$level} commission from user ID: " . $user->id,
                'amount' => $commission,
                'credit' => $commission,
                'type' => 'credit',
                'status' => 'approved',
                'date' => now(),
            ]);

            $currentUser = $upline;
        }
    }

    /* --- ADMINISTRATIVE METHODS --- */

    public function index_recharge()
    {
        $datas = Recharge::orderByDesc('id')->get();
        return view('admin.pages.list', compact('datas'));
    }

    public function index_recharge_status(Request $request, $id)
    {
        $data = Recharge::where('id', $id)->first();
        if ($request->status == 'rejected') {
            $user = User::where('id', $data->user_id)->first();
            $user->balance = $user->balance + $data->amount;
            $user->save();
        }

        $data->status = $request->status;
        $data->save();
        return redirect()->back()->with('success', 'Status changed successfully.');
    }

    public function login()
    {
        if (Auth::guard('admin')->check()) {
            return redirect()->route('admin.dashboard');
        }
        return view('admin.auth.login');
    }

    public function login_submit(Request $request)
    {
        $credentials = $request->only('email', 'password');
        if (Auth::guard('admin')->attempt($credentials)) {
            $admin = Auth::guard('admin')->user();
            if ($admin->type == 'admin') {
                return redirect()->route('admin.dashboard')->with('success', 'Logged In Successfully.');
            } else {
                return redirect()->route('admin.login')->with('error', 'Unauthorized access.');
            }
        } else {
            return redirect()->route('admin.login')->with('error', 'Invalid admin credentials.');
        }
    }

    public function logout()
    {
        if (Auth::guard('admin')->check()) {
            Auth::guard('admin')->logout();
            return redirect()->route('admin.login')->with('success', 'Logged out successfully.');
        } else {
            return redirect()->route('admin.login')->with('error', 'You are already logged out.');
        }
    }

    public function dashboard() { return view('admin.dashboard'); }

    public function profile() { return view('admin.profile.index'); }

    public function profile_update()
    {
        $admin = Admin::first();
        return view('admin.profile.update-details', compact('admin'));
    }

    public function profile_update_submit(Request $request)
    {
        $admin = Admin::find(1);
        $path = uploadImage(false, $request, 'photo', 'admin/assets/images/profile/', $admin->photo);
        $admin->photo = $path ?? $admin->photo;
        $admin->name = $request->name;
        $admin->email = $request->email;
        $admin->phone = $request->phone;
        $admin->address = $request->address;
        $admin->update();
        return redirect()->route('admin.profile.update')->with('success', 'Admin profile updated.');
    }

    public function change_password()
    {
        $admin = Auth::guard('admin')->user();
        return view('admin.profile.change-password', compact('admin'));
    }

    public function check_password(Request $request)
    {
        $admin = Auth::guard('admin')->user();
        $password = $request->password;
        if (Hash::check($password, $admin->password)) {
            return response()->json(['message' => 'Password matched.', 'status' => true]);
        } else {
            return response()->json(['message' => 'Password does not match.', 'status' => false]);
        }
    }

    public function change_password_submit(Request $request)
    {
        $validate = Validator::make($request->all(), [
            'old_password' => 'required',
            'new_password' => 'required|min:6',
            'confirm_password' => 'required|same:new_password'
        ]);

        if ($validate->fails()) {
            return redirect()->route('admin.changepassword')->withErrors($validate->errors());
        }

        $admin = Auth::guard('admin')->user();
        if (Hash::check($request->old_password, $admin->password)) {
            $admin->password = Hash::make($request->new_password);
            $admin->update();
            return redirect()->route('admin.changepassword')->with('success', 'Password updated successfully');
        } else {
            return redirect()->route('admin.changepassword')->with('error', 'Current password is incorrect.');
        }
    }
}
